'use client';

import React, { useState, useEffect } from 'react';
import {
  ArrowLeft,
  Save,
  User,
  Mail,
  Phone,
  Building2,
  Shield,
  AlertCircle,
  Eye,
  EyeOff
} from 'lucide-react';
import Link from 'next/link';
import { useRouter, useParams } from 'next/navigation';
import type { Department, UserRole, AgencyPermission, AgencyUser } from '@/types';
import { agencyUserApi } from '@/services/api';
import PermissionSelector from '@/components/permissions/PermissionSelector';

interface FormData {
  firstName: string;
  lastName: string;
  email: string;
  phone: string;
  departmentId: string;
  role: UserRole;
  permissions: AgencyPermission[];
  isActive: boolean;
}

export default function EditUserPage() {
  const router = useRouter();
  const params = useParams();
  const userId = params?.id as string;

  const [departments, setDepartments] = useState<Department[]>([]);
  const [loading, setLoading] = useState(false);
  const [loadingData, setLoadingData] = useState(true);
  const [errors, setErrors] = useState<Partial<FormData>>({});

  const [formData, setFormData] = useState<FormData>({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    departmentId: '',
    role: 'specialist',
    permissions: [],
    isActive: true
  });

  useEffect(() => {
    if (userId) {
      fetchUser();
      fetchDepartments();
    }
  }, [userId]);

  const fetchUser = async () => {
    try {
      setLoadingData(true);
      const data = await agencyUserApi.getAgencyUser(userId);

      setFormData({
        firstName: data.first_name,
        lastName: data.last_name,
        email: data.email,
        phone: data.phone || '',
        departmentId: data.department_id,
        role: data.role,
        permissions: data.permissions ? (typeof data.permissions === 'string' ? JSON.parse(data.permissions) : data.permissions) : [],
        isActive: Boolean(data.is_active)
      });
    } catch (error) {
      console.error('Error fetching user:', error);
      alert('Kullanıcı bilgileri yüklenirken bir hata oluştu');
    } finally {
      setLoadingData(false);
    }
  };

  const fetchDepartments = async () => {
    try {
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/departments`);
      const data = await response.json();

      const departmentsWithDates = data.map((dept: Record<string, unknown>) => ({
        id: dept.id,
        name: dept.name,
        description: dept.description || '',
        color: dept.color,
        parentDepartmentId: dept.parent_department_id,
        managerId: dept.manager_id,
        isActive: Boolean(dept.is_active),
        userCount: dept.user_count || 0,
        permissions: dept.permissions ? (typeof dept.permissions === 'string' ? JSON.parse(dept.permissions) : dept.permissions) : [],
        createdAt: new Date(dept.created_at),
        updatedAt: new Date(dept.updated_at)
      }));

      setDepartments(departmentsWithDates);
    } catch (error) {
      console.error('Error fetching departments:', error);
    }
  };

  const handleInputChange = (field: keyof FormData, value: string | boolean | AgencyPermission[]) => {
    setFormData(prev => ({ ...prev, [field]: value }));

    // Departman değiştiğinde, o departmanın yetkilerini otomatik olarak ata
    if (field === 'departmentId' && typeof value === 'string' && value) {
      const selectedDept = departments.find(d => d.id === value);
      if (selectedDept && selectedDept.permissions) {
        setFormData(prev => ({
          ...prev,
          departmentId: value,
          permissions: selectedDept.permissions as AgencyPermission[]
        }));
      }
    }

    // Clear error when user starts typing
    if (errors[field]) {
      setErrors(prev => ({ ...prev, [field]: undefined }));
    }
  };

  const validateForm = (): boolean => {
    const newErrors: Partial<FormData> = {};

    if (!formData.firstName.trim()) {
      newErrors.firstName = 'Ad gereklidir';
    }

    if (!formData.lastName.trim()) {
      newErrors.lastName = 'Soyad gereklidir';
    }

    if (!formData.email.trim()) {
      newErrors.email = 'E-posta gereklidir';
    } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
      newErrors.email = 'Geçerli bir e-posta adresi giriniz';
    }

    if (!formData.departmentId) {
      newErrors.departmentId = 'Departman seçimi gereklidir';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!validateForm()) {
      return;
    }

    try {
      setLoading(true);

      // Update user via API
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/agency-users/${userId}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          first_name: formData.firstName,
          last_name: formData.lastName,
          email: formData.email,
          phone: formData.phone,
          department_id: formData.departmentId,
          role: formData.role,
          permissions: formData.permissions,
          is_active: formData.isActive
        })
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to update user');
      }

      alert('Kullanıcı başarıyla güncellendi');
      router.push(`/users/${userId}`);
    } catch (error) {
      console.error('Error updating user:', error);
      alert((error as Error).message || 'Kullanıcı güncellenirken bir hata oluştu. Lütfen tekrar deneyin.');
    } finally {
      setLoading(false);
    }
  };

  const roleOptions = [
    { value: 'intern', label: 'Stajyer' },
    { value: 'assistant', label: 'Uzman Yardımcısı' },
    { value: 'specialist', label: 'Uzman' },
    { value: 'senior_specialist', label: 'Kıdemli Uzman' },
    { value: 'department_manager', label: 'Departman Yöneticisi' },
    { value: 'agency_manager', label: 'Acente Yöneticisi' },
    { value: 'system_admin', label: 'Sistem Yöneticisi' }
  ];

  if (loadingData) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-4">
          <Link
            href={`/users/${userId}`}
            className="flex items-center text-gray-600 hover:text-gray-900"
          >
            <ArrowLeft className="h-5 w-5 mr-2" />
            Kullanıcıya Dön
          </Link>
          <div className="h-6 border-l border-gray-300"></div>
          <h1 className="text-2xl font-bold text-gray-900">Kullanıcıyı Düzenle</h1>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="max-w-3xl">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-6">
          <h2 className="text-lg font-medium text-gray-900 flex items-center">
            <User className="h-5 w-5 mr-2" />
            Kişisel Bilgiler
          </h2>

          {/* Ad Soyad */}
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Ad *
              </label>
              <input
                type="text"
                value={formData.firstName}
                onChange={(e) => handleInputChange('firstName', e.target.value)}
                className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                  errors.firstName ? 'border-red-300' : 'border-gray-300'
                }`}
                placeholder="Örn: Ahmet"
              />
              {errors.firstName && (
                <p className="mt-1 text-sm text-red-600 flex items-center">
                  <AlertCircle className="h-4 w-4 mr-1" />
                  {errors.firstName}
                </p>
              )}
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Soyad *
              </label>
              <input
                type="text"
                value={formData.lastName}
                onChange={(e) => handleInputChange('lastName', e.target.value)}
                className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                  errors.lastName ? 'border-red-300' : 'border-gray-300'
                }`}
                placeholder="Örn: Yılmaz"
              />
              {errors.lastName && (
                <p className="mt-1 text-sm text-red-600 flex items-center">
                  <AlertCircle className="h-4 w-4 mr-1" />
                  {errors.lastName}
                </p>
              )}
            </div>
          </div>

          {/* İletişim */}
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                E-posta *
              </label>
              <input
                type="email"
                value={formData.email}
                onChange={(e) => handleInputChange('email', e.target.value)}
                className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                  errors.email ? 'border-red-300' : 'border-gray-300'
                }`}
                placeholder="ornek@email.com"
              />
              {errors.email && (
                <p className="mt-1 text-sm text-red-600 flex items-center">
                  <AlertCircle className="h-4 w-4 mr-1" />
                  {errors.email}
                </p>
              )}
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Telefon
              </label>
              <input
                type="tel"
                value={formData.phone}
                onChange={(e) => handleInputChange('phone', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="+90 555 123 45 67"
              />
            </div>
          </div>

          {/* Departman */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Departman *
            </label>
            <select
              value={formData.departmentId}
              onChange={(e) => handleInputChange('departmentId', e.target.value)}
              className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent ${
                errors.departmentId ? 'border-red-300' : 'border-gray-300'
              }`}
            >
              <option value="">Departman seçiniz</option>
              {departments.filter(d => d.isActive).map((dept) => (
                <option key={dept.id} value={dept.id}>
                  {dept.name}
                </option>
              ))}
            </select>
            {errors.departmentId && (
              <p className="mt-1 text-sm text-red-600 flex items-center">
                <AlertCircle className="h-4 w-4 mr-1" />
                {errors.departmentId}
              </p>
            )}
          </div>

          {/* Rol */}
          <div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Rol *
              </label>
              <select
                value={formData.role}
                onChange={(e) => handleInputChange('role', e.target.value as UserRole)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              >
                {roleOptions.map((role) => (
                  <option key={role.value} value={role.value}>
                    {role.label}
                  </option>
                ))}
              </select>
            </div>
          </div>

          {/* Yetkiler */}
          <div className="border-t border-gray-200 pt-6">
            <h3 className="text-sm font-medium text-gray-700 mb-3">Kullanıcı Yetkileri</h3>
            <PermissionSelector
              selectedPermissions={formData.permissions}
              onChange={(perms) => handleInputChange('permissions', perms)}
              departmentName={departments.find(d => d.id === formData.departmentId)?.name}
            />
          </div>

          {/* Durum */}
          <div className="border-t border-gray-200 pt-6">
            <label className="flex items-center">
              <input
                type="checkbox"
                checked={formData.isActive}
                onChange={(e) => handleInputChange('isActive', e.target.checked)}
                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <span className="ml-2 text-sm text-gray-700">Kullanıcı aktif</span>
            </label>
            <p className="mt-1 text-xs text-gray-500">
              Pasif kullanıcılar sisteme giriş yapamaz ve departman listelerinde görünmez.
            </p>
          </div>

          {/* Submit Button */}
          <div className="flex justify-end space-x-3 pt-6 border-t border-gray-200">
            <Link
              href={`/users/${userId}`}
              className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50"
            >
              İptal
            </Link>
            <button
              type="submit"
              disabled={loading}
              className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {loading ? (
                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
              ) : (
                <Save className="h-4 w-4 mr-2" />
              )}
              {loading ? 'Kaydediliyor...' : 'Değişiklikleri Kaydet'}
            </button>
          </div>
        </div>
      </form>
    </div>
  );
}
